<?php
/**
 * Admin area settings and hooks.
 *
 * @package DR_PRO
 * @subpackage  DR_PRO
 */

namespace DR_PRO;

defined( 'ABSPATH' ) || exit;

/**
 * Global Settings.
 */
class Delicious_Recipes_Pro_Admin {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Initialization.
	 *
	 * @return void
	 * @since 1.0.0
	 * @access private
	 */
	private function init() {

		// Initialize hooks.
		$this->init_hooks();

		// Allow 3rd party to remove hooks.
		do_action( 'wp_delicious_admin_unhook', $this );
	}

	/**
	 * Initialize hooks.
	 *
	 * @return void
	 * @since 1.0.0
	 * @access private
	 */
	private function init_hooks() {
		// Admin Scripts.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		// Admin Script Translations.
		add_action( 'enqueue_block_editor_assets', array( $this, 'get_editor_assets' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'set_script_translations' ), 99999999999 );

		add_filter( 'delicious_recipes_global_settings_defaults', array( $this, 'global_settings_defaults' ) );

		// register post type.
		add_action( 'init', array( $this, 'register_post_types' ) );

		// register taxonomy.
		add_action( 'init', array( $this, 'register_taxonomies' ) );

		// Add meta box.
		add_action( 'add_meta_boxes', array( $this, 'add_new_equipment_metabox' ) );

		// Add ADMIN COLUMN - HEADERS.
		add_filter( 'manage_edit-equipment_columns', array( $this, 'equipment_columns' ) );

		// ADMIN COLUMN - Featured CONTENT.
		add_action( 'manage_equipment_posts_custom_column', array( $this, 'featured_equipments' ), 10, 2 );

		// Add Submissions(count) view in recipe.
		add_filter( 'views_edit-recipe', array( $this, 'add_submissions_view' ), 10, 1 );

		// Add custom query var.
		add_filter( 'query_vars', array( $this, 'add_meta_query_vars' ) );

		// Manipulate query.
		add_action( 'pre_get_posts', array( $this, 'pre_get_posts_submission_query' ) );

		// Populate the recipe submission structure option.
		add_action( 'init', array( $this, 'get_recipe_submission_structure' ) );

		// Add custom index page.
		add_action( 'init', array( $this, 'custom_blog_redirect' ) );
	}

	/**
	 * Redirect to custom recipe index page.
	 *
	 * @since v2.2.0
	 */
	function custom_blog_redirect() {
		$global_settings = delicious_recipes_get_global_settings();
		$recipe_base     = isset( $global_settings['recipeBase'] ) && ! empty( $global_settings['recipeBase'] ) ? trim( $global_settings['recipeBase'], '/\\' ) : 'recipe';
		if ( preg_match( '#/' . preg_quote( $recipe_base, '#' ) . '/?$#', $_SERVER['REQUEST_URI'] ) ) {
			if ( isset( $global_settings['recipeIndexPage'] ) && isset( $global_settings['recipeIndexPage'][0] ) && 'yes' === ( $global_settings['recipeIndexPage'][0] ) ) {
				if ( isset( $global_settings['customRecipeIndexPage'] ) && '' !== ( $global_settings['customRecipeIndexPage'] ) ) {
					$customRecipeIndexPage = $global_settings['customRecipeIndexPage'];
					wp_redirect( get_permalink( $customRecipeIndexPage ) );
					exit;
				}
			}
		}
	}

	/**
	 * Enqueue Admin Scripts
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

		$screen         = get_current_screen();
		$global_deps    = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/global.asset.php';
		$recipe_deps    = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/recipe.asset.php';
		$equipment_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipment.asset.php';
		$analytics_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/analytics.asset.php';

		// License Statuses Global value.
		$license_details = get_option( 'delicious_recipes_pro_license' );
		$license_status  = '';
		if ( isset( $license_details ) && ! empty( $license_details ) ) {
			$license_status = isset( $license_details['license_status'] ) ? $license_details['license_status'] : '';
		}
		wp_enqueue_media();

		wp_enqueue_style( 'delicious-recipes-pro-admin', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/admin/css/delicious-recipes-pro-admin.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/admin/css/delicious-recipes-pro-admin.css' ), 'all' );
		wp_enqueue_style( 'delicious-recipes-pro-admin-styles', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/adminCSS.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/adminCSS.css' ), 'all' );

		wp_enqueue_script( 'delicious-recipes-pro-admin', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/admin/js/delicious-recipes-pro-admin.js', array( 'jquery' ), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/admin/js/delicious-recipes-pro-admin.js' ), true );
		wp_localize_script(
			'delicious-recipes-pro-admin',
			'DeliciousRecipesProAdmin',
			array(
				'ajaxURL'       => esc_url( admin_url( 'admin-ajax.php' ) ),
				'nonce'         => wp_create_nonce( 'delicious_recipes_pro_nonce' ),
				'licenseStatus' => ( 'valid' === $license_status ) ? true : false,
			)
		);

		if ( isset( $screen->id ) && is_string( $screen->id ) && strpos( $screen->id ?? '', '_page_delicious_recipes_global_settings' ) > 0 ) {

			// Recipe global screen assets.
			wp_register_script( 'delicious-recipes-pro-global-settings', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/global.js', $global_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/global.js' ), true );

			$default_templates = array(
				'newRecipeSubmissionContent' => delicious_recipes_pro_get_template_content( 'new_submission', 'emails/new-recipe-submission.php', 'customer', true ),
				'recipePublishedContent'     => delicious_recipes_pro_get_template_content( 'recipe_published', 'emails/recipe-published.php', 'customer', true ),
			);

			global $submenu;
			foreach ( $submenu[''] as $submenu_item ) {
				if ( 'delicious_recipes_recipe_submission' === $submenu_item[2] ) {
					$submenu_slug = $submenu_item[2];
					$submenu_href = admin_url( 'admin.php?page=' . $submenu_slug );  // Generate the full URL.
				}
			}
			// Add localization vars.
			wp_localize_script(
				'delicious-recipes-pro-global-settings',
				'DeliciousRecipesPro',
				array(
					'siteURL'                 => esc_url( rtrim( home_url(), '/' ) ),
					'pluginUrl'               => esc_url( plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) ),
					'defaultTemplates'        => $default_templates,
					'recipeSubmissionMenuURL' => esc_url( $submenu_href ),
					'ajaxURL'                 => esc_url( admin_url( 'admin-ajax.php' ) ),
					'licenseNonce'            => wp_create_nonce( 'delicious_recipes_pro_nonce' ),
					'licenseStatus'           => 'valid' === $license_status ? true : false,
					'restNonce'               => wp_create_nonce( 'wp_rest' ), // Rest Nonce for global recipe index additional settings.
				)
			);
			wp_enqueue_script( 'delicious-recipes-pro-global-settings' );
		}

		if ( $screen->post_type == DELICIOUS_RECIPES_EQUIPMENT_POST_TYPE ) {
			// Equipment screen assets.
			wp_register_script( 'delicious-recipes-pro-equipment-settings', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipment.js', $equipment_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/equipment.js' ), true );

			// Add localization vars.
			wp_localize_script(
				'delicious-recipes-pro-equipment-settings',
				'DeliciousRecipesPro',
				array(
					'siteURL'   => esc_url( rtrim( home_url(), '/' ) ),
					'pluginUrl' => esc_url( plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) ),
				)
			);
			wp_enqueue_script( 'delicious-recipes-pro-equipment-settings' );

			wp_enqueue_style( 'toastr', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.css' ), 'all' );
			wp_enqueue_script( 'toastr', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.js', array( 'jquery' ), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.js' ), true );
		}

		if ( $screen->post_type == DELICIOUS_RECIPE_POST_TYPE ) {
			// recipe screen assets.
			wp_register_script( 'delicious-recipes-pro-recipe-settings', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/recipe.js', $recipe_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/recipe.js' ), true );

			wp_enqueue_style( 'delicious-recipes-nutrition-translation', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/singleRecipeCSS.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/singleRecipeCSS.css' ), 'all' );

			// Add localization vars.
			wp_localize_script(
				'delicious-recipes-pro-recipe-settings',
				'DeliciousRecipeSettings',
				array(
					'siteURL'            => esc_url( rtrim( home_url(), '/' ) ),
					'pluginUrl'          => esc_url( plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) ),
					'globalSettings'     => delicious_recipes_get_global_settings(),
					'AIAssistantEnabled' => function_exists( 'WP_DEL_AI_RECIPE_ASSISTANT' ),
					'licenseStatus'      => 'valid' === $license_status ? true : false,
				)
			);
			wp_enqueue_script( 'delicious-recipes-pro-recipe-settings' );
		}

		if ( isset( $screen->id ) && is_string( $screen->id ) && strpos( $screen->id ?? '', '_page_delicious_recipes_analytics_dashboard' ) > 0 ) {

			wp_enqueue_style( 'delicious-recipes-pro-analytics', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/admin/css/delicious-recipes-pro-analytics.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/admin/css/delicious-recipes-pro-analytics.css' ), 'all' );

			wp_register_script( 'delicious-recipes-analytics', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/analytics.js', $analytics_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/analytics.js' ), true );

			// Add localization vars.
			wp_localize_script(
				'delicious-recipes-analytics',
				'DeliciousRecipesAnalytics',
				array(
					'ajaxURL'       => esc_url( admin_url( 'admin-ajax.php' ) ),
					'siteURL'       => esc_url( rtrim( home_url(), '/' ) ),
					'adminURL'      => esc_url( admin_url() ),
					'pluginUrl'     => esc_url( plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) ),
					'greetings'     => array(
						'morning'   => __( 'Good Morning', 'delicious-recipes-pro' ),
						'afternoon' => __( 'Good Afternoon', 'delicious-recipes-pro' ),
						'evening'   => __( 'Good Evening', 'delicious-recipes-pro' ),
					),
					'licenseStatus' => ( $license_status === 'valid' ) ? true : false,
				)
			);
			wp_enqueue_script( 'delicious-recipes-analytics' );
		}

		if ( isset( $screen->id ) && is_string( $screen->id ) && strpos( $screen->id ?? '', '_page_delicious_recipes_reviews' ) > 0 ) {

			$reviews_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/reviews.asset.php';

			wp_register_script( 'delicious-recipes-reviews', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/reviews.js', $reviews_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/reviews.js' ), true );

			// Add localization vars.
			wp_localize_script(
				'delicious-recipes-reviews',
				'DeliciousRecipesReviews',
				array(
					'ajaxURL'        => esc_url( admin_url( 'admin-ajax.php' ) ),
					'siteURL'        => esc_url( rtrim( home_url(), '/' ) ),
					'adminURL'       => esc_url( admin_url() ),
					'pluginUrl'      => esc_url( plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) ),
					'userID'         => get_current_user_id(),
					'allRecipes'     => delicious_recipes_pro_get_all_recipes(),
					'globalSettings' => delicious_recipes_get_global_settings(),
				)
			);

			wp_enqueue_script( 'delicious-recipes-reviews' );

		}

		if ( isset( $screen->id ) && is_string( $screen->id ) && strpos( $screen->id ?? '', '_page_delicious_recipes_auto_link_ingredients' ) > 0 ) {
			$auto_link_ingredients_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/autoLinkIngredients.asset.php';
			$license_details            = get_option( 'delicious_recipes_pro_license' );
			$license_status             = '';
			if ( isset( $license_details ) && ! empty( $license_details ) ) {
				$license_status = isset( $license_details['license_status'] ) ? $license_details['license_status'] : '';
			}
			wp_register_script( 'delicious-recipes-ingredient-links', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/autoLinkIngredients.js', $auto_link_ingredients_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/autoLinkIngredients.js' ), true );
			wp_localize_script(
				'delicious-recipes-ingredient-links',
				'deliciousRecipesIngredientLinks',
				array(
					'pluginURL'     => plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ),
					'siteURL'       => get_site_url(),
					'nonce'         => wp_create_nonce( 'delicious_recipes_auto_link_ingredients' ),
					'licenseStatus' => ( $license_status === 'valid' ) ? true : false,
				)
			);
			wp_enqueue_script( 'delicious-recipes-ingredient-links' );
			wp_enqueue_style( 'delicious-recipes-ingredient-links', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/autoLinkIngredients.css', array( 'wp-components' ), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/autoLinkIngredients.css' ), 'all' );
			wp_enqueue_style( 'toastr', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.css' ), 'all' );
			wp_enqueue_script( 'toastr', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.js', array( 'jquery' ), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.js' ), true );

		}

		if ( isset( $screen->id ) && is_string( $screen->id ) && strpos( $screen->id ?? '', '_page_delicious_recipes_recipe_submission' ) > 0 ) {
			$recipe_submission_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/recipeSubmissionMenu.asset.php';
			wp_register_script( 'delicious-recipes-recipe-submission', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/recipeSubmissionMenu.js', $recipe_submission_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/recipeSubmissionMenu.js' ), true );
			$img = esc_url( plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/images/Delicious-Recipes.png' );
			wp_localize_script(
				'delicious-recipes-recipe-submission',
				'deliciousRecipeSubmission',
				array(
					'imageURL' => $img,
				)
			);
			wp_enqueue_script( 'delicious-recipes-recipe-submission' );
			wp_enqueue_style( 'toastr', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.css', array(), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.css' ), 'all' );
			wp_enqueue_script( 'toastr', plugin_dir_url( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.js', array( 'jquery' ), filemtime( plugin_dir_path( DELICIOUS_RECIPES_PLUGIN_FILE ) . 'assets/lib/toastr/toastr.min.js' ), true );
		}

		if ( isset( $screen->id ) && strpos( $screen->id, '_page_delicious_recipes_whats_new' ) > 0 ) {
			$whats_new_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/whatsNewChangelog.asset.php';
			wp_register_script( 'wp-del-pro-whats-new', plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/whatsNewChangelog.js', $whats_new_deps['dependencies'], filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/whatsNewChangelog.js' ), true );
			wp_localize_script(
				'wp-del-pro-whats-new',
				'deliciousRecipesWhatsNew',
				array(
					'pluginURL' => plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ),
					'siteURL'   => get_site_url(),
				)
			);
			wp_enqueue_script( 'wp-del-pro-whats-new' );
		}
	}


	/**
	 * Enqueue Block Editor Assets.
	 */
	public function get_editor_assets() {
		$blocks_deps = include_once plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/blocksJS.asset.php';
		// Scripts.
		wp_enqueue_script(
			'delicious-recipes-pro-block-js', // Handle.
			plugin_dir_url( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/blocksJS.js',
			$blocks_deps['dependencies'],
			filemtime( plugin_dir_path( DELICIOUS_RECIPES_PRO_PLUGIN_FILE ) . 'assets/build/blocksJS.js' ),
			true
		);
	}

	/**
	 * Set Script Translations
	 *
	 * @return void
	 */
	public function set_script_translations() {
		wp_set_script_translations( 'delicious-recipes-pro-global-settings', 'delicious-recipes-pro' ); // Global Settings.
		wp_set_script_translations( 'delicious-recipes-pro-recipe-settings', 'delicious-recipes-pro' ); // Recipe Settings.
		wp_set_script_translations( 'delicious-recipes-pro-equipment-settings', 'delicious-recipes-pro' ); // Equipment Settings.
		wp_set_script_translations( 'delicious-recipes-analytics', 'delicious-recipes-pro' ); // Analytics Settings.
		wp_set_script_translations( 'delicious-recipes-reviews', 'delicious-recipes-pro' ); // Review Listings.
		wp_set_script_translations( 'delicious-recipes-pro-block-js', 'delicious-recipes-pro' ); // Block Editor.
		wp_set_script_translations( 'delicious-recipes-ingredient-links', 'delicious-recipes-pro' ); // Ingredient Links.
	}

	/**
	 * Get default global settings values.
	 *
	 * @return void
	 */
	public function global_settings_defaults( $global_defaults = array() ) {

		$pro_defaults = apply_filters(
			'delicious_recipes_global_settings_pro_defaults',
			array(
				// Additional Customization fields.
				'bannerLayouts'                    => array(
					'0' => array(
						'id'                            => 'default',
						'backgroundColor'               => '#f6f3ef',
						'textColor'                     => '#000000',
						'enableBannerBreadcrumb'        => false,
						'enableBannerNutritionalValues' => false,
						'enableBannerRecipeKeys'        => false,
						'enableBannerRecipeInfo'        => true,
					),
					'1' => array(
						'id'                            => 'layout-1',
						'backgroundColor'               => '#f6f3ef',
						'textColor'                     => '#333333',
						'enableBannerBreadcrumb'        => true,
						'enableBannerNutritionalValues' => true,
						'enableBannerRecipeKeys'        => true,
						'enableBannerRecipeInfo'        => true,
					),
					'2' => array(
						'id'                            => 'layout-2',
						'backgroundColor'               => '#f6f3ef',
						'textColor'                     => '#000000',
						'enableBannerBreadcrumb'        => true,
						'enableBannerNutritionalValues' => true,
						'enableBannerRecipeKeys'        => true,
						'enableBannerRecipeInfo'        => true,
					),
					'3' => array(
						'id'                            => 'layout-3',
						'backgroundColor'               => '#f6f3ef',
						'textColor'                     => '#000000',
						'enableBannerBreadcrumb'        => true,
						'enableBannerNutritionalValues' => true,
						'enableBannerRecipeKeys'        => true,
						'enableBannerRecipeInfo'        => true,
					),
					'4' => array(
						'id'                            => 'layout-4',
						'backgroundColor'               => '#f6f3ef',
						'textColor'                     => '#000000',
						'enableBannerBreadcrumb'        => true,
						'enableBannerNutritionalValues' => true,
						'enableBannerRecipeKeys'        => true,
						'enableBannerRecipeInfo'        => true,
					),
					'5' => array(
						'id'                            => 'layout-5',
						'backgroundColor'               => '#f5f5f5',
						'textColor'                     => '#000000',
						'enableBannerBreadcrumb'        => true,
						'enableBannerNutritionalValues' => true,
						'enableBannerRecipeKeys'        => true,
						'enableBannerRecipeInfo'        => true,
					),
				),
				'bannerLayoutId'                   => 'default',
				'selectedBannerLayout'             => array(
					'id'    => 'default',
					'label' => 'Default',
				),
				// Additional Ingredients fields.
				'displayUnitConversion'            => array(),
				'defaultUnitSystem'                => 'usCustomary',
				'showIngredientsImage'             => array(),

				'enableRecipeSubmission'           => array(),
				'recipeSubmissionGuideline'        => '',
				'recipeDisclaimer'                 => '',
				'enableNotification'               => array(
					'0' => 'yes',
				),
				'emailAddresses'                   => '',
				'enablePendingMode'                => array(
					'0' => 'yes',
				),
				'enableGuestMode'                  => array(
					'0' => 'yes',
				),
				'allowDeleteRecipe'                => array(),
				'recaptchaEnabledForRecipe'        => array(),
				// Instructions fields.
				'enableMultipleInstructionImages'  => array(),
				// Email Templates fields.
				'newRecipeSubmissionSubject'       => __( 'Your recipe submission has been received!', 'delicious-recipes-pro' ),
				'newRecipeSubmissionContent'       => '',
				'recipePublishedSubject'           => __( 'Your recipe submission has been published!', 'delicious-recipes-pro' ),
				'recipePublishedContent'           => '',

				'recipeFieldsOrder'                => array(
					'0' => array(
						'id'           => 'recipe-details',
						'sectionTitle' => __( 'Recipe Details', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
					'1' => array(
						'id'           => 'recipe-info',
						'sectionTitle' => __( 'Recipe Card Information', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
					'2' => array(
						'id'           => 'recipe-ingredients',
						'sectionTitle' => __( 'Ingredients', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
					'3' => array(
						'id'           => 'recipe-instructions',
						'sectionTitle' => __( 'Instruction', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
					'4' => array(
						'id'           => 'recipe-gallery',
						'sectionTitle' => __( 'Gallery', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
					'5' => array(
						'id'           => 'recipe-nutrition',
						'sectionTitle' => __( 'Nutrition Facts', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
					'6' => array(
						'id'           => 'recipe-notes',
						'sectionTitle' => __( 'Notes', 'delicious-recipes-pro' ),
						'enable'       => array( '0' => 'yes' ),
					),
				),
				'enableAffiliateLinkIndicator'     => array(),
				'enableCookMode'                   => array(
					'0' => 'yes',
				),
				'enableFloatingBar'                => array(
					'0' => 'yes',
				),
				'floatingBarToggles'               => array(
					'0' => array(
						'label'  => __( 'Start Reading Mode', 'delicious-recipes-pro' ),
						'key'    => 'startReadingMode',
						'id'     => 'dr-reading-mode',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'1' => array(
						'label'  => __( 'Start Timer', 'delicious-recipes-pro' ),
						'key'    => 'startTimer',
						'id'     => 'dr-start-timer',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'2' => array(
						'label'  => __( 'Share', 'delicious-recipes-pro' ),
						'key'    => 'socialShare',
						'id'     => 'dr-social-share',
						'enable' => array(
							'0' => 'yes',
						),
					),
				),
				'enableEquipments'                 => array(
					'0' => 'yes',
				),
				'enableEquipmentLayoutEdit'        => array(),
				'equipmentLayout'                  => 'grid',
				'equipmentPerRow'                  => 3,
				'enableQuestions'                  => array(
					'0' => 'yes',
				),
				'enableRatingsWithoutComment'      => array(
					'0' => 'yes',
				),
				'ratingForceComment'               => '1',
				'enableReviewImages'               => array(
					'0' => 'yes',
				),
				'relatedRecipes'                   => array(),
				'relatedSectionTitle'              => __( 'Related Recipes', 'delicious-recipes-pro' ),
				'relatedFilter'                    => 'recipe-course',
				'showThumbnail'                    => array(
					'0' => 'yes',
				),
				'fullwidthRelatedSection'          => array(),
				'showAfterParagraph'               => 3,
				'noofRelatedRecipes'               => 3,
				'autoloadRecipes'                  => array(),
				'autoloadPerPage'                  => 4,
				'autoloadFilter'                   => 'prev',
				'affiliateDisclosure'              => '',
				'enableDisclosureSingle'           => array(),
				'disclosureLocation'               => 'top',
				'enableDisclosureEquipment'        => array(),
				'enableDisclosureRecipeCard'       => array(),
				'emojiReactions'                   => array(),
				'enableUniqueReactions'            => array(
					'0' => 'yes',
				),
				'emojiSectionTitle'                => __( "What's Your Reaction?", 'delicious-recipes-pro' ),
				'emojiReactionsList'               => array(
					'0'  => array(
						'label'  => __( 'Reaction - Yummy', 'delicious-recipes-pro' ),
						'id'     => 'yummy',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'1'  => array(
						'label'  => __( 'Reaction - Love', 'delicious-recipes-pro' ),
						'id'     => 'love',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'2'  => array(
						'label'  => __( 'Reaction - Wow', 'delicious-recipes-pro' ),
						'id'     => 'wow',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'3'  => array(
						'label'  => __( 'Reaction - Happy', 'delicious-recipes-pro' ),
						'id'     => 'happy',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'4'  => array(
						'label'  => __( 'Reaction - Sad', 'delicious-recipes-pro' ),
						'id'     => 'sad',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'5'  => array(
						'label'  => __( 'Reaction - Yay', 'delicious-recipes-pro' ),
						'id'     => 'yay',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'6'  => array(
						'label'  => __( 'Reaction - Rolf', 'delicious-recipes-pro' ),
						'id'     => 'rolf',
						'enable' => array(),
					),
					'7'  => array(
						'label'  => __( 'Reaction - Smiling', 'delicious-recipes-pro' ),
						'id'     => 'smiling',
						'enable' => array(),
					),
					'8'  => array(
						'label'  => __( 'Reaction - Sleepy', 'delicious-recipes-pro' ),
						'id'     => 'sleepy',
						'enable' => array(),
					),
					'9'  => array(
						'label'  => __( 'Reaction - Angry', 'delicious-recipes-pro' ),
						'id'     => 'angry',
						'enable' => array(),
					),
					'10' => array(
						'label'  => __( 'Reaction - Dead', 'delicious-recipes-pro' ),
						'id'     => 'dead',
						'enable' => array(
							'0' => 'yes',
						),
					),
					'11' => array(
						'label'  => __( 'Reaction - Wink', 'delicious-recipes-pro' ),
						'id'     => 'wink',
						'enable' => array(),
					),
				),
				'emojiIconSize'                    => '48',
				'emojiLocation'                    => 'center',

				// for ratings review.
				'enableWouldYouRecommend'          => array(),
				'enableDidYouMake'                 => array(),
				'enableReviewTags'                 => array(),
				'reviewTags'                       => array(),
				'wouldYouRecommendLabel'           => __( 'Would you recommend this recipe?', 'delicious-recipes-pro' ),
				'didYouMakeLabel'                  => __( 'Did you make this recipe?', 'delicious-recipes-pro' ),
				'enableExitIntent'                 => array(),

				// for collapsible nutrition chart.
				'enableCollapsibleNutritionChart'  => array(),
				'collapsibleNutritionChartLabel'   => __( 'Show Full Nutrition Label', 'delicious-recipes-pro' ),

				// for nutrition calculator.
				'enableAutoNutritionCalculator'    => array(),
				'defaultAnalysisAPI'               => '',
				'nutritionCalculatorAppId'         => '',
				'nutritionCalculatorAppKey'        => '',
				'spoonacularApiKey'                => '',

				// for nutrition translation.
				'enableAutoIngredientsTranslation' => array(),
				'ingredientsTranslationApiKey'     => '',

				// for CTA.
				'globalCTAImage'                   => array(),
				'globalCTALink'                    => '',
				'globalCTAOpenInNewTab'            => array(),
				'globalCTAAttributes'              => array(),

				// for recipe index page.
				'recipeIndexPage'                  => array(),
				'customRecipeIndexPage'            => '',

				// for Dashboard Portal.
				'dashboardPortal'                  => array(),
				'dashboardPortalLogo'              => '',
				'dashboardPortalLogoPreview'       => '',
				'dashboardTabs'                    => array(),
			)
		);

		$global_defaults = array_merge( $global_defaults, $pro_defaults );

		return $global_defaults;
	}

	/**
	 * Add equipment post type.
	 *
	 * @return void
	 * @since 1.0.0
	 * @access public
	 */
	public function register_post_types() {

		// Post Type labels.
		$labels = array(
			'name'               => _x( 'Equipment', 'post type general name', 'delicious-recipes-pro' ),
			'singular_name'      => _x( 'Equipment', 'post type singular name', 'delicious-recipes-pro' ),
			'menu_name'          => _x( 'Equipment', 'admin menu', 'delicious-recipes-pro' ),
			'name_admin_bar'     => _x( 'Equipment', 'add new on admin bar', 'delicious-recipes-pro' ),
			'add_new'            => _x( 'Add New', 'Equipment', 'delicious-recipes-pro' ),
			'add_new_item'       => __( 'Add New Equipment', 'delicious-recipes-pro' ),
			'new_item'           => __( 'New Equipment', 'delicious-recipes-pro' ),
			'edit_item'          => __( 'Edit Equipment', 'delicious-recipes-pro' ),
			'view_item'          => __( 'View Equipment', 'delicious-recipes-pro' ),
			'all_items'          => __( 'Equipment', 'delicious-recipes-pro' ),
			'search_items'       => __( 'Search Equipment', 'delicious-recipes-pro' ),
			'parent_item_colon'  => __( 'Parent Equipment:', 'delicious-recipes-pro' ),
			'not_found'          => __( 'No Equipment found.', 'delicious-recipes-pro' ),
			'not_found_in_trash' => __( 'No Equipment found in Trash.', 'delicious-recipes-pro' ),
		);

		$args = array(
			'labels'             => $labels,
			'description'        => __( 'Description.', 'delicious-recipes-pro' ),
			'public'             => false,
			'menu_icon'          => '',
			'publicly_queryable' => false,
			'show_ui'            => true,
			'show_in_menu'       => 'delicious-recipes',
			'show_in_rest'       => true,
			'query_var'          => true,
			'rewrite'            => array(
				'slug'       => 'equipment',
				'with_front' => true,
			),
			'capability_type'    => 'post',
			'has_archive'        => true,
			'hierarchical'       => false,
			'menu_position'      => 30,
			'supports'           => array( 'title', 'editor', 'thumbnail' ),
		);

		register_post_type( DELICIOUS_RECIPES_EQUIPMENT_POST_TYPE, $args );

		if ( 'yes' === get_option( 'delicious_recipes_queue_flush_rewrite_rules' ) ) {
			update_option( 'delicious_recipes_queue_flush_rewrite_rules', 'no' );
			flush_rewrite_rules();
		}
	}

	/**
	 * Register Taxonomies.
	 *
	 * @return void
	 */
	public function register_taxonomies() {

		$taxonomy_labels = array(
			'name'              => _x( 'Equipment Categories', 'taxonomy general name', 'delicious-recipes-pro' ),
			'singular_name'     => _x( 'Equipment Category', 'taxonomy singular name', 'delicious-recipes-pro' ),
			'search_items'      => __( 'Search Equipment Categories', 'delicious-recipes-pro' ),
			'all_items'         => __( 'All Equipment Categories', 'delicious-recipes-pro' ),
			'parent_item'       => __( 'Parent Equipment Category', 'delicious-recipes-pro' ),
			'parent_item_colon' => __( 'Parent Equipment Category:', 'delicious-recipes-pro' ),
			'edit_item'         => __( 'Edit Equipment Category', 'delicious-recipes-pro' ),
			'update_item'       => __( 'Update Equipment Category', 'delicious-recipes-pro' ),
			'add_new_item'      => __( 'Add New Equipment Category', 'delicious-recipes-pro' ),
			'new_item_name'     => __( 'New Equipment Category Name', 'delicious-recipes-pro' ),
			'menu_name'         => __( 'Equipment Categories', 'delicious-recipes-pro' ),
		);

		$taxonomy_args = array(
			'hierarchical'      => false,
			'labels'            => $taxonomy_labels,
			'show_ui'           => true,
			'show_in_rest'      => true,
			'show_admin_column' => true,
			'rewrite'           => array(
				'slug'         => 'equipment-category',
				'hierarchical' => false,
			),
		);

		register_taxonomy( 'equipment-category', array( DELICIOUS_RECIPES_EQUIPMENT_POST_TYPE ), $taxonomy_args );
	}

	/**
	 * Add new equipment metabox.
	 *
	 * @return void
	 */
	public function add_new_equipment_metabox() {
		add_meta_box(
			'delicious_recipes_equipment_metabox',
			__( 'Equipment Settings', 'delicious-recipes-pro' ),
			array( $this, 'equipment_mb_callback' ),
			DELICIOUS_RECIPES_EQUIPMENT_POST_TYPE,
			'normal',
			'high'
		);
	}

	/**
	 * Equipment Metabox Callback.
	 *
	 * @return void
	 */
	public function equipment_mb_callback( $post ) {
		?>
		<div id="delicious-recipe-equipment" data-rest-nonce="<?php echo wp_create_nonce( 'wp_rest' ); ?>"
			data-post-id="<?php echo esc_attr( $post->ID ); ?>"></div>
		<?php
	}

	/**
	 * Customize Admin column.
	 *
	 * @param Array $equipment_columns List of columns.
	 *
	 * @return Array                  [description]
	 */
	public function equipment_columns( $equipment_columns ) {
		$equipment_columns['featured'] = __( 'Featured', 'delicious-recipes-pro' );

		return $equipment_columns;
	}

	/**
	 * Add data to custom column.
	 *
	 * @param String $column_name Custom column name.
	 * @param int    $id Post ID.
	 */
	public function featured_equipments( $column_name, $id ) {
		switch ( $column_name ) {
			case 'featured':
				$featured = get_post_meta( $id, 'wp_delicious_featured_equipment', true );
				$featured = ( isset( $featured ) && '' != $featured ) ? $featured : 'no';

				$icon_class = ' dashicons-star-empty ';
				if ( ! empty( $featured ) && 'yes' === $featured ) {
					$icon_class = ' dashicons-star-filled ';
				}
				$nonce = wp_create_nonce( 'wp_delicious_featured_equipment_nonce' );
				printf( '<a href="#" class="dr-featured-equipment dashicons %s" data-post-id="%d"  data-nonce="%s"></a>', $icon_class, $id, $nonce );
				break;
			default:
				break;
		} // end switch
	}

	/**
	 * Renders all of the submission recipes in 'All Recipes' page of WordPress.
	 *
	 * @return views array with 'Submissions' view.
	 */
	public function add_submissions_view( $views ) {
		$view_name = 'Submissions';

		// Build the anchor for the 'Submissions' view and push it to the $views array.
		$views[] = '
				<a href="' . add_query_arg(
			array(
				'post_type'   => DELICIOUS_RECIPE_POST_TYPE,
				'post_status' => 'all',
				'submission'  => 1,
			),
			'edit.php'
		) . '" ' . delicious_recipes_pro_get_submission_attributes() . '>' . $view_name . ' <span class="count">(' . delicious_recipes_pro_filter_submission_results( true ) . ')</span></a>';

		return $views;
	}

	/**
	 * Adds submission as query_vars.
	 */
	public function add_meta_query_vars( $vars ) {
		$vars[] = 'submission';

		return $vars;
	}

	/**
	 * Determines if we're looking at submision query_var and sets meta query.
	 */
	public function pre_get_posts_submission_query( $query ) {
		if ( is_admin() && $query->is_main_query() && $query->is_post_type_archive( DELICIOUS_RECIPE_POST_TYPE ) && isset( $_GET['submission'] ) ) {
			$query->set(
				'meta_query',
				array(
					array(
						'key'   => '_dr_user_submission_recipe',
						'value' => 'yes',
					),
				)
			);
		}
	}

	/**
	 * Populate the recipe submission structure option in init hook to avoid issues in user dashboard page in frontend.
	 *
	 * @return void
	 */
	public function get_recipe_submission_structure() {
		$recipe_submission_structure        = array(
			'enabledItems' => array(
				'recipeTitle'        => array( true, true ),
				'recipeImage'        => array( true, true ),
				'description'        => array( true, false ),
				'cookingMethods'     => array( true, false ),
				'cuisine'            => array( true, false ),
				'courses'            => array( true, false ),
				'recipeKeys'         => array( true, false ),
				'recipeBadge'        => array( true, false ),
				'recipeDietary'      => array( true, false ),
				'recipeSubtitle'     => array( true, false ),
				'recipeDescription'  => array( true, false ),
				'recipeKeywords'     => array( true, false ),
				'difficultyLevel'    => array( true, true ),
				'preparationTime'    => array( true, true ),
				'cookingTime'        => array( true, true ),
				'restTime'           => array( true, true ),
				'calories'           => array( true, false ),
				'bestSeason'         => array( true, false ),
				'cookingTemp'        => array( true, false ),
				'estimatedCost'      => array( true, false ),
				'noOfServings'       => array( true, true ),
				'ingredientsSection' => array( true, true ),
				'instructionSection' => array( true, true ),
				'imageGallery'       => array( true, false ),
				'videoGallery'       => array( true, false ),
				'nutritionFacts'     => array( true, false ),
				'faqs'               => array( true, false ),
				'notes'              => array( true, false ),
				'notesToAdmin'       => array( true, false ),
			),
			'fieldsOrder'  => array(
				'recipeDetailsItems' => array(
					array(
						'id'      => 'recipe-title',
						'name'    => __( 'Recipe Title', 'delicious-recipes-pro' ),
						'content' => 'recipeTitle',
					),
					array(
						'id'      => 'recipe-image',
						'name'    => __( 'Recipe Featured Image', 'delicious-recipes-pro' ),
						'content' => 'recipeImage',
					),
					array(
						'id'      => 'description',
						'name'    => __( 'Description', 'delicious-recipes-pro' ),
						'content' => 'description',
					),
					array(
						'id'      => 'cooking-methods',
						'name'    => __( 'Cooking Methods', 'delicious-recipes-pro' ),
						'content' => 'cookingMethods',
					),
					array(
						'id'      => 'cuisine',
						'name'    => __( 'Cuisines', 'delicious-recipes-pro' ),
						'content' => 'cuisine',
					),
					array(
						'id'      => 'courses',
						'name'    => __( 'Courses', 'delicious-recipes-pro' ),
						'content' => 'courses',
					),
					array(
						'id'      => 'recipe-keys',
						'name'    => __( 'Recipe Keys', 'delicious-recipes-pro' ),
						'content' => 'recipeKeys',
					),
					array(
						'id'      => 'recipe-badge',
						'name'    => __( 'Recipe Badge', 'delicious-recipes-pro' ),
						'content' => 'recipeBadge',
					),
					array(
						'id'      => 'recipe-dietary',
						'name'    => __( 'Recipe Dietary', 'delicious-recipes-pro' ),
						'content' => 'recipeDietary',
					),
				),
				'recipeCardItems'    => array(
					array(
						'id'      => 'recipe-subtitle',
						'name'    => __( 'Recipe Subtitle', 'delicious-recipes-pro' ),
						'content' => 'recipeSubtitle',
					),
					array(
						'id'      => 'recipe-description',
						'name'    => __( 'Recipe Description', 'delicious-recipes-pro' ),
						'content' => 'recipeDescription',
					),
					array(
						'id'      => 'recipe-keywords',
						'name'    => __( 'Recipe Keywords', 'delicious-recipes-pro' ),
						'content' => 'recipeKeywords',
					),
					array(
						'id'      => 'difficulty-level',
						'name'    => __( 'Difficulty Level', 'delicious-recipes-pro' ),
						'content' => 'difficultyLevel',
					),
					array(
						'id'      => 'preparation-time',
						'name'    => __( 'Prep Time', 'delicious-recipes-pro' ),
						'content' => 'preparationTime',
					),
					array(
						'id'      => 'cooking-time',
						'name'    => __( 'Cook Time', 'delicious-recipes-pro' ),
						'content' => 'cookingTime',
					),
					array(
						'id'      => 'rest-time',
						'name'    => __( 'Rest Time', 'delicious-recipes-pro' ),
						'content' => 'restTime',
					),
					array(
						'id'      => 'calories',
						'name'    => __( 'Calories', 'delicious-recipes-pro' ),
						'content' => 'calories',
					),
					array(
						'id'      => 'best-season',
						'name'    => __( 'Best Season', 'delicious-recipes-pro' ),
						'content' => 'bestSeason',
					),
					array(
						'id'      => 'cooking-temp',
						'name'    => __( 'Cooking Temp', 'delicious-recipes-pro' ),
						'content' => 'cookingTemp',
					),
					array(
						'id'      => 'estimated-cost',
						'name'    => __( 'Estimated Cost', 'delicious-recipes-pro' ),
						'content' => 'estimatedCost',
					),
				),
				'ingredientsItems'   => array(
					array(
						'id'      => 'no-of-servings',
						'name'    => __( 'No. of Servings', 'delicious-recipes-pro' ),
						'content' => 'noOfServings',
					),
					array(
						'id'      => 'ingredients-section',
						'name'    => __( 'Ingredients', 'delicious-recipes-pro' ),
						'content' => 'ingredientsSection',
					),
				),
				'instructionsItems'  => array(
					array(
						'id'      => 'instruction-section',
						'name'    => __( 'Instructions', 'delicious-recipes-pro' ),
						'content' => 'instructionSection',
					),
				),
				'galleryItems'       => array(
					array(
						'id'      => 'recipe-gallery',
						'name'    => __( 'Image Gallery', 'delicious-recipes-pro' ),
						'content' => 'imageGallery',
					),
					array(
						'id'      => 'video-gallery',
						'name'    => __( 'Video Gallery', 'delicious-recipes-pro' ),
						'content' => 'videoGallery',
					),
				),
				'otherItems'         => array(
					array(
						'id'      => 'nutrition-facts',
						'name'    => __( 'Nutrition Info', 'delicious-recipes-pro' ),
						'content' => 'nutritionFacts',
					),
					array(
						'id'      => 'notes',
						'name'    => __( 'Notes', 'delicious-recipes-pro' ),
						'content' => 'notes',
					),
					array(
						'id'      => 'notes-to-admin',
						'name'    => __( 'Notes to Admin', 'delicious-recipes-pro' ),
						'content' => 'notesToAdmin',
					),
				),
			),
		);
		$recipe_submission_structure_option = get_option( 'recipe_submission_structure' );

		if ( ! $recipe_submission_structure_option || ! isset( $recipe_submission_structure_option ) ) {
			update_option( 'recipe_submission_structure', $recipe_submission_structure );
		} else {
			$recipe_submission_structure = $recipe_submission_structure_option;
		}

		update_option( 'recipe_submission_structure', $recipe_submission_structure );
	}
}

new Delicious_Recipes_Pro();
